<?php
header('Content-Type: application/json; charset=UTF-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

// Prevent caching
header('Cache-Control: no-cache, no-store, must-revalidate');
header('Pragma: no-cache');
header('Expires: 0');

// Error handling
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Get POST data
$input = @file_get_contents('php://input');
$data = @json_decode($input, true);

if ($data === null && json_last_error() !== JSON_ERROR_NONE) {
    $data = $_POST;
}

if (!$data || !isset($data['platform']) || !isset($data['url'])) {
    echo json_encode([
        'success' => false,
        'message' => 'Invalid request. Platform and URL are required.'
    ]);
    exit;
}

$platform = strtolower(trim($data['platform']));
$url = trim($data['url']);

// Validate platform
$validPlatforms = ['facebook', 'youtube', 'instagram'];
if (!in_array($platform, $validPlatforms)) {
    echo json_encode([
        'success' => false,
        'message' => 'Invalid platform specified.'
    ]);
    exit;
}

// Process based on platform
try {
    switch ($platform) {
        case 'facebook':
            $result = downloadFacebook($url);
            break;
        case 'youtube':
            $result = downloadYoutube($url);
            break;
        case 'instagram':
            $result = downloadInstagram($url);
            break;
        default:
            $result = [
                'success' => false,
                'message' => 'Platform not supported yet.'
            ];
    }
    
    echo json_encode($result);
} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'message' => 'An error occurred: ' . $e->getMessage()
    ]);
}

// Facebook Download Function
function downloadFacebook($url) {
    // NOTE: Facebook video downloading requires proper API access or third-party services
    // This is a placeholder implementation. You'll need to integrate with services like:
    // - Facebook Graph API (requires access token)
    // - Third-party APIs like RapidAPI's Facebook Video Downloader
    
    // For demonstration, returning mock data
    // In production, you would make API calls here
    
    if (!preg_match('/facebook\.com|fb\.watch/i', $url)) {
        return [
            'success' => false,
            'message' => 'Invalid Facebook URL'
        ];
    }
    
    // Example API integration (you need to implement actual API calls)
    // $apiResponse = callFacebookDownloadAPI($url);
    
    // Mock response for demonstration
    return [
        'success' => true,
        'video' => [
            'title' => 'Facebook Video',
            'thumbnail' => 'https://via.placeholder.com/640x360?text=Facebook+Video',
            'duration' => '5:30',
            'author' => 'Sample User',
            'qualities' => [
                [
                    'quality' => 'HD 1080p',
                    'size' => '45.2 MB',
                    'url' => '#' // Replace with actual download URL from API
                ],
                [
                    'quality' => 'HD 720p',
                    'size' => '28.5 MB',
                    'url' => '#'
                ],
                [
                    'quality' => 'SD 480p',
                    'size' => '15.3 MB',
                    'url' => '#'
                ]
            ]
        ]
    ];
}

// YouTube Download Function
function downloadYoutube($url) {
    // NOTE: YouTube downloading requires compliance with their Terms of Service
    // Consider using official YouTube API or approved third-party services
    // Popular options:
    // - youtube-dl or yt-dlp (command-line tools)
    // - Third-party APIs (RapidAPI, etc.)
    
    if (!preg_match('/youtube\.com|youtu\.be/i', $url)) {
        return [
            'success' => false,
            'message' => 'Invalid YouTube URL'
        ];
    }
    
    // Extract video ID
    preg_match('/(?:youtube\.com\/watch\?v=|youtu\.be\/)([^&\s]+)/', $url, $matches);
    $videoId = $matches[1] ?? null;
    
    if (!$videoId) {
        return [
            'success' => false,
            'message' => 'Could not extract video ID'
        ];
    }
    
    // Mock response for demonstration
    // In production, integrate with YouTube API or download service
    return [
        'success' => true,
        'video' => [
            'title' => 'YouTube Video Title',
            'thumbnail' => "https://img.youtube.com/vi/{$videoId}/maxresdefault.jpg",
            'duration' => '10:45',
            'author' => 'Channel Name',
            'views' => '1.2M views',
            'qualities' => [
                [
                    'quality' => '4K 2160p',
                    'size' => '850 MB',
                    'url' => '#'
                ],
                [
                    'quality' => 'Full HD 1080p',
                    'size' => '320 MB',
                    'url' => '#'
                ],
                [
                    'quality' => 'HD 720p',
                    'size' => '180 MB',
                    'url' => '#'
                ],
                [
                    'quality' => 'SD 480p',
                    'size' => '95 MB',
                    'url' => '#'
                ],
                [
                    'quality' => 'Audio Only (MP3)',
                    'size' => '12 MB',
                    'url' => '#'
                ]
            ]
        ]
    ];
}

// Instagram Download Function
function downloadInstagram($url) {
    // NOTE: Instagram downloading requires proper authentication or third-party services
    // Options:
    // - Instagram Basic Display API
    // - Third-party scraping services (check their ToS)
    
    if (!preg_match('/instagram\.com/i', $url)) {
        return [
            'success' => false,
            'message' => 'Invalid Instagram URL'
        ];
    }
    
    // Mock response for demonstration
    return [
        'success' => true,
        'video' => [
            'title' => 'Instagram Video/Reel',
            'thumbnail' => 'https://via.placeholder.com/640x640?text=Instagram+Video',
            'duration' => '0:30',
            'author' => '@username',
            'qualities' => [
                [
                    'quality' => 'HD Quality',
                    'size' => '8.5 MB',
                    'url' => '#'
                ],
                [
                    'quality' => 'Standard Quality',
                    'size' => '4.2 MB',
                    'url' => '#'
                ]
            ]
        ]
    ];
}

// Helper function to make API calls (example)
function callAPI($url, $method = 'GET', $data = null, $headers = []) {
    $ch = curl_init();
    
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    
    if ($method === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        if ($data) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }
    }
    
    if (!empty($headers)) {
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    }
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    return [
        'code' => $httpCode,
        'response' => $response
    ];
}

// Sanitize filename
function sanitizeFilename($filename) {
    $filename = preg_replace('/[^a-zA-Z0-9\-_\.]/', '_', $filename);
    return substr($filename, 0, 200);
}

// Log downloads (optional - for analytics)
function logDownload($platform, $url, $success) {
    $logFile = 'logs/downloads.log';
    $logDir = dirname($logFile);
    
    if (!file_exists($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    $logEntry = sprintf(
        "[%s] Platform: %s | URL: %s | Success: %s | IP: %s\n",
        date('Y-m-d H:i:s'),
        $platform,
        $url,
        $success ? 'YES' : 'NO',
        $_SERVER['REMOTE_ADDR'] ?? 'Unknown'
    );
    
    file_put_contents($logFile, $logEntry, FILE_APPEND);
}
?>
