// Preloader
window.addEventListener('load', () => {
    const preloader = document.getElementById('preloader');
    if (preloader) {
        setTimeout(() => {
            preloader.classList.add('hidden');
        }, 500);
    }
});

// Navbar scroll effect
const navbar = document.querySelector('.navbar');
if (navbar) {
    window.addEventListener('scroll', () => {
        if (window.scrollY > 50) {
            navbar.classList.add('scrolled');
        } else {
            navbar.classList.remove('scrolled');
        }
    });
}

// Counter animation for hero stats
const animateCounter = (element, target, duration = 2000) => {
    const start = 0;
    const increment = target / (duration / 16);
    let current = start;
    
    const timer = setInterval(() => {
        current += increment;
        if (current >= target) {
            element.textContent = target.toLocaleString();
            clearInterval(timer);
        } else {
            element.textContent = Math.floor(current).toLocaleString();
        }
    }, 16);
};

// Trigger counter animation when hero is visible
const observeCounters = () => {
    const counters = document.querySelectorAll('.stat-number');
    
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting && entry.target.textContent === '0') {
                const target = parseInt(entry.target.getAttribute('data-target'));
                animateCounter(entry.target, target);
                observer.unobserve(entry.target);
            }
        });
    }, { threshold: 0.5 });
    
    counters.forEach(counter => observer.observe(counter));
};

if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', observeCounters);
} else {
    observeCounters();
}

// Back to Top Button
const backToTopBtn = document.getElementById('backToTop');
if (backToTopBtn) {
    window.addEventListener('scroll', () => {
        if (window.scrollY > 300) {
            backToTopBtn.classList.add('show');
        } else {
            backToTopBtn.classList.remove('show');
        }
    });
    
    backToTopBtn.addEventListener('click', () => {
        window.scrollTo({
            top: 0,
            behavior: 'smooth'
        });
    });
}

// Toast Notification System
class ToastNotification {
    constructor() {
        this.container = this.createContainer();
    }
    
    createContainer() {
        let container = document.querySelector('.toast-container');
        if (!container) {
            container = document.createElement('div');
            container.className = 'toast-container';
            document.body.appendChild(container);
        }
        return container;
    }
    
    show(message, type = 'info', duration = 5000) {
        const toast = document.createElement('div');
        toast.className = `toast ${type}`;
        
        const icons = {
            success: '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor"><path d="M22 11.08V12a10 10 0 1 1-5.93-9.14" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/><polyline points="22 4 12 14.01 9 11.01" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/></svg>',
            error: '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor"><circle cx="12" cy="12" r="10" stroke-width="2"/><line x1="15" y1="9" x2="9" y2="15" stroke-width="2" stroke-linecap="round"/><line x1="9" y1="9" x2="15" y2="15" stroke-width="2" stroke-linecap="round"/></svg>',
            warning: '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor"><path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/><line x1="12" y1="9" x2="12" y2="13" stroke-width="2" stroke-linecap="round"/><line x1="12" y1="17" x2="12.01" y2="17" stroke-width="2" stroke-linecap="round"/></svg>',
            info: '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor"><circle cx="12" cy="12" r="10" stroke-width="2"/><line x1="12" y1="16" x2="12" y2="12" stroke-width="2" stroke-linecap="round"/><line x1="12" y1="8" x2="12.01" y2="8" stroke-width="2" stroke-linecap="round"/></svg>'
        };
        
        toast.innerHTML = `
            <div class="toast-icon">${icons[type]}</div>
            <div class="toast-content">
                <div class="toast-message">${message}</div>
            </div>
            <button class="toast-close" onclick="this.parentElement.remove()">
                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                    <line x1="18" y1="6" x2="6" y2="18" stroke-width="2" stroke-linecap="round"/>
                    <line x1="6" y1="6" x2="18" y2="18" stroke-width="2" stroke-linecap="round"/>
                </svg>
            </button>
        `;
        
        this.container.appendChild(toast);
        
        if (duration > 0) {
            setTimeout(() => {
                toast.classList.add('removing');
                setTimeout(() => toast.remove(), 300);
            }, duration);
        }
        
        return toast;
    }
    
    success(message, duration) {
        return this.show(message, 'success', duration);
    }
    
    error(message, duration) {
        return this.show(message, 'error', duration);
    }
    
    warning(message, duration) {
        return this.show(message, 'warning', duration);
    }
    
    info(message, duration) {
        return this.show(message, 'info', duration);
    }
}

const toast = new ToastNotification();

// Mobile Menu Toggle
const mobileMenuToggle = document.querySelector('.mobile-menu-toggle');
const navMenu = document.querySelector('.nav-menu');

if (mobileMenuToggle) {
    mobileMenuToggle.addEventListener('click', () => {
        navMenu.classList.toggle('active');
    });
}

// Platform Selector Buttons
const platformBtns = document.querySelectorAll('.platform-btn');
const downloaderSections = document.querySelectorAll('.downloader-section');

platformBtns.forEach(btn => {
    btn.addEventListener('click', () => {
        const platform = btn.getAttribute('data-platform');
        
        // Update active button
        platformBtns.forEach(b => b.classList.remove('active'));
        btn.classList.add('active');
        
        // Show corresponding section
        downloaderSections.forEach(section => {
            section.classList.remove('active');
        });
        
        const targetSection = document.getElementById(platform);
        if (targetSection) {
            targetSection.classList.add('active');
            targetSection.scrollIntoView({ behavior: 'smooth', block: 'start' });
        }
    });
});

// Navigation Links
const navLinks = document.querySelectorAll('.nav-link');

navLinks.forEach(link => {
    link.addEventListener('click', (e) => {
        e.preventDefault();
        const targetId = link.getAttribute('href').substring(1);
        
        // Update active nav link
        navLinks.forEach(l => l.classList.remove('active'));
        link.classList.add('active');
        
        // Handle platform sections
        if (targetId === 'facebook' || targetId === 'youtube' || targetId === 'instagram') {
            // Update platform button
            platformBtns.forEach(btn => {
                if (btn.getAttribute('data-platform') === targetId) {
                    btn.classList.add('active');
                } else {
                    btn.classList.remove('active');
                }
            });
            
            // Show section
            downloaderSections.forEach(section => {
                section.classList.remove('active');
            });
            
            const targetSection = document.getElementById(targetId);
            if (targetSection) {
                targetSection.classList.add('active');
            }
        }
        
        // Scroll to section
        const target = document.getElementById(targetId);
        if (target) {
            target.scrollIntoView({ behavior: 'smooth', block: 'start' });
        }
        
        // Close mobile menu
        if (navMenu.classList.contains('active')) {
            navMenu.classList.remove('active');
        }
    });
});

// Download Video Function
async function downloadVideo(platform) {
    const urlInput = document.getElementById(`${platform}-url`);
    const resultContainer = document.getElementById(`${platform}-result`);
    const downloadBtn = document.querySelector(`.${platform}-btn`);
    
    const url = urlInput.value.trim();
    
    // Validate URL
    if (!url) {
        toast.warning('Please enter a video URL');
        urlInput.focus();
        return;
    }
    
    if (!isValidUrl(url, platform)) {
        toast.error(`Please enter a valid ${platform} video URL`);
        urlInput.focus();
        return;
    }
    
    // Show loading state
    downloadBtn.classList.add('loading');
    downloadBtn.disabled = true;
    resultContainer.innerHTML = '';
    resultContainer.classList.remove('show');
    
    // Show processing toast
    const processingToast = toast.info('Processing your request...', 0);
    
    try {
        // Send request to PHP backend
        const response = await fetch('download.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                platform: platform,
                url: url
            })
        });
        
        const data = await response.json();
        
        // Remove loading state
        downloadBtn.classList.remove('loading');
        downloadBtn.disabled = false;
        processingToast.remove();
        
        if (data.success) {
            showResult(resultContainer, data.video, platform);
            toast.success('Video information retrieved successfully!');
            
            // Smooth scroll to results
            resultContainer.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
        } else {
            showError(resultContainer, data.message || 'Failed to fetch video. Please try again.');
            toast.error(data.message || 'Failed to fetch video');
        }
    } catch (error) {
        downloadBtn.classList.remove('loading');
        downloadBtn.disabled = false;
        processingToast.remove();
        showError(resultContainer, 'An error occurred. Please try again later.');
        toast.error('Connection error. Please check your internet and try again.');
        console.error('Error:', error);
    }
}

// Validate URL
function isValidUrl(url, platform) {
    const patterns = {
        facebook: /^(https?:\/\/)?(www\.)?(facebook\.com|fb\.watch)\/.+/i,
        youtube: /^(https?:\/\/)?(www\.)?(youtube\.com|youtu\.be)\/.+/i,
        instagram: /^(https?:\/\/)?(www\.)?instagram\.com\/.+/i
    };
    
    return patterns[platform] && patterns[platform].test(url);
}

// Show Error Message
function showError(container, message) {
    container.innerHTML = `
        <div class="error-message">
            <svg style="width: 24px; height: 24px; display: inline-block; vertical-align: middle; margin-right: 8px;" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <circle cx="12" cy="12" r="10" stroke-width="2"/>
                <line x1="12" y1="8" x2="12" y2="12" stroke-width="2" stroke-linecap="round"/>
                <line x1="12" y1="16" x2="12.01" y2="16" stroke-width="2" stroke-linecap="round"/>
            </svg>
            <strong>Error:</strong> ${message}
        </div>
    `;
    container.classList.add('show');
}

// Show Success Result
function showResult(container, video, platform) {
    let qualityOptionsHtml = '';
    
    if (video.qualities && video.qualities.length > 0) {
        qualityOptionsHtml = '<div class="quality-options">';
        video.qualities.forEach(quality => {
            qualityOptionsHtml += `
                <button class="quality-btn" onclick="downloadFile('${quality.url}', '${video.title}')">
                    <span class="quality-label">${quality.quality}</span>
                    <span class="quality-size">${quality.size || 'N/A'}</span>
                </button>
            `;
        });
        qualityOptionsHtml += '</div>';
    }
    
    container.innerHTML = `
        <div class="result-card">
            <div class="video-info">
                ${video.thumbnail ? `<img src="${video.thumbnail}" alt="Video thumbnail" class="video-thumbnail">` : ''}
                <div class="video-details">
                    <h3>${video.title || 'Video'}</h3>
                    <div class="video-meta">
                        ${video.duration ? `<span>⏱️ ${video.duration}</span>` : ''}
                        ${video.author ? `<span>👤 ${video.author}</span>` : ''}
                        ${video.views ? `<span>👁️ ${video.views}</span>` : ''}
                    </div>
                </div>
            </div>
            ${qualityOptionsHtml}
            ${!qualityOptionsHtml && video.download_url ? `
                <button class="download-btn ${platform}-btn" onclick="downloadFile('${video.download_url}', '${video.title}')">
                    <svg class="btn-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                        <path d="M21 15V19C21 19.5304 20.7893 20.0391 20.4142 20.4142C20.0391 20.7893 19.5304 21 19 21H5C4.46957 21 3.96086 20.7893 3.58579 20.4142C3.21071 20.0391 3 19.5304 3 19V15" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                        <path d="M7 10L12 15L17 10" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                        <path d="M12 15V3" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                    </svg>
                    Download Now
                </button>
            ` : ''}
        </div>
    `;
    container.classList.add('show');
}

// Download File
function downloadFile(url, filename) {
    const a = document.createElement('a');
    a.href = url;
    a.download = filename || 'video';
    a.target = '_blank';
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
}

// Smooth Scroll for Anchor Links
document.querySelectorAll('a[href^="#"]').forEach(anchor => {
    anchor.addEventListener('click', function (e) {
        const href = this.getAttribute('href');
        if (href !== '#' && href !== '#home') {
            e.preventDefault();
            const target = document.querySelector(href);
            if (target) {
                target.scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
            }
        }
    });
});

// Input validation on paste
document.querySelectorAll('.video-url-input').forEach(input => {
    input.addEventListener('paste', (e) => {
        setTimeout(() => {
            const url = input.value.trim();
            if (url) {
                input.style.borderColor = 'var(--success-color)';
            }
        }, 100);
    });
    
    input.addEventListener('input', (e) => {
        if (!input.value.trim()) {
            input.style.borderColor = '';
        }
    });
});

// Lazy load AdSense (for better page speed)
window.addEventListener('load', () => {
    const adsenseScript = document.querySelectorAll('script[src*="adsbygoogle"]');
    adsenseScript.forEach(script => {
        if (script.getAttribute('data-ad-client')) {
            (window.adsbygoogle = window.adsbygoogle || []).push({});
        }
    });
});

// Tool cards interaction
const toolCards = document.querySelectorAll('.tool-card');
toolCards.forEach(card => {
    card.addEventListener('click', () => {
        alert('This feature is coming soon! Stay tuned.');
    });
});
