<?php
/**
 * Utility Functions for VideoDownloader
 * Common helper functions used throughout the application
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    define('ABSPATH', dirname(__FILE__) . '/');
}

// Include configuration
if (file_exists(dirname(__FILE__) . '/config.php')) {
    require_once dirname(__FILE__) . '/config.php';
}

/**
 * Validate video URL based on platform
 */
function validateUrl($url, $platform) {
    $pattern = getUrlPattern($platform);
    if (!$pattern) {
        return false;
    }
    return preg_match($pattern, $url) === 1;
}

/**
 * Extract video ID from YouTube URL
 */
function extractYouTubeId($url) {
    $patterns = [
        '/youtube\.com\/watch\?v=([^&\s]+)/',
        '/youtu\.be\/([^?\s]+)/',
        '/youtube\.com\/embed\/([^?\s]+)/',
        '/youtube\.com\/v\/([^?\s]+)/'
    ];
    
    foreach ($patterns as $pattern) {
        if (preg_match($pattern, $url, $matches)) {
            return $matches[1];
        }
    }
    
    return null;
}

/**
 * Extract Facebook video ID
 */
function extractFacebookId($url) {
    if (preg_match('/\/videos\/(\d+)/', $url, $matches)) {
        return $matches[1];
    }
    return null;
}

/**
 * Sanitize filename for download
 */
function sanitizeFilename($filename) {
    // Remove special characters
    $filename = preg_replace('/[^a-zA-Z0-9\-_\.]/', '_', $filename);
    // Remove multiple underscores
    $filename = preg_replace('/_+/', '_', $filename);
    // Limit length
    return substr($filename, 0, 200);
}

/**
 * Format file size
 */
function formatFileSize($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    } else {
        return $bytes . ' bytes';
    }
}

/**
 * Format duration (seconds to HH:MM:SS)
 */
function formatDuration($seconds) {
    $hours = floor($seconds / 3600);
    $minutes = floor(($seconds % 3600) / 60);
    $seconds = $seconds % 60;
    
    if ($hours > 0) {
        return sprintf('%d:%02d:%02d', $hours, $minutes, $seconds);
    } else {
        return sprintf('%d:%02d', $minutes, $seconds);
    }
}

/**
 * Get client IP address
 */
function getClientIP() {
    $ipaddress = '';
    
    if (isset($_SERVER['HTTP_CLIENT_IP'])) {
        $ipaddress = $_SERVER['HTTP_CLIENT_IP'];
    } elseif (isset($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        $ipaddress = $_SERVER['HTTP_X_FORWARDED_FOR'];
    } elseif (isset($_SERVER['HTTP_X_FORWARDED'])) {
        $ipaddress = $_SERVER['HTTP_X_FORWARDED'];
    } elseif (isset($_SERVER['HTTP_FORWARDED_FOR'])) {
        $ipaddress = $_SERVER['HTTP_FORWARDED_FOR'];
    } elseif (isset($_SERVER['HTTP_FORWARDED'])) {
        $ipaddress = $_SERVER['HTTP_FORWARDED'];
    } elseif (isset($_SERVER['REMOTE_ADDR'])) {
        $ipaddress = $_SERVER['REMOTE_ADDR'];
    } else {
        $ipaddress = 'UNKNOWN';
    }
    
    return $ipaddress;
}

/**
 * Rate limiting check
 */
function checkRateLimit($ip, $maxRequests = 10, $timeWindow = 60) {
    if (!ENABLE_RATE_LIMITING) {
        return true;
    }
    
    $cacheFile = 'cache/rate_limit_' . md5($ip) . '.json';
    $cacheDir = dirname($cacheFile);
    
    if (!file_exists($cacheDir)) {
        mkdir($cacheDir, 0755, true);
    }
    
    $now = time();
    $requests = [];
    
    // Load existing requests
    if (file_exists($cacheFile)) {
        $data = json_decode(file_get_contents($cacheFile), true);
        $requests = $data['requests'] ?? [];
    }
    
    // Remove old requests outside time window
    $requests = array_filter($requests, function($timestamp) use ($now, $timeWindow) {
        return ($now - $timestamp) < $timeWindow;
    });
    
    // Check if limit exceeded
    if (count($requests) >= $maxRequests) {
        return false;
    }
    
    // Add current request
    $requests[] = $now;
    
    // Save updated requests
    file_put_contents($cacheFile, json_encode(['requests' => $requests]));
    
    return true;
}

/**
 * Log download to file
 */
function logDownload($platform, $url, $success = true, $error = null) {
    if (!ENABLE_LOGGING) {
        return;
    }
    
    $logDir = 'logs';
    if (!file_exists($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    $logFile = $logDir . '/downloads_' . date('Y-m-d') . '.log';
    
    $logEntry = sprintf(
        "[%s] IP: %s | Platform: %s | URL: %s | Success: %s%s\n",
        date('Y-m-d H:i:s'),
        getClientIP(),
        $platform,
        $url,
        $success ? 'YES' : 'NO',
        $error ? ' | Error: ' . $error : ''
    );
    
    file_put_contents($logFile, $logEntry, FILE_APPEND);
}

/**
 * Cache video data
 */
function cacheVideoData($key, $data, $duration = 3600) {
    if (!ENABLE_CACHE) {
        return false;
    }
    
    $cacheDir = 'cache';
    if (!file_exists($cacheDir)) {
        mkdir($cacheDir, 0755, true);
    }
    
    $cacheFile = $cacheDir . '/' . md5($key) . '.json';
    
    $cacheData = [
        'data' => $data,
        'expires' => time() + $duration
    ];
    
    return file_put_contents($cacheFile, json_encode($cacheData)) !== false;
}

/**
 * Get cached video data
 */
function getCachedVideoData($key) {
    if (!ENABLE_CACHE) {
        return null;
    }
    
    $cacheFile = 'cache/' . md5($key) . '.json';
    
    if (!file_exists($cacheFile)) {
        return null;
    }
    
    $cacheData = json_decode(file_get_contents($cacheFile), true);
    
    if (!$cacheData || $cacheData['expires'] < time()) {
        // Cache expired
        unlink($cacheFile);
        return null;
    }
    
    return $cacheData['data'];
}

/**
 * Make HTTP request with cURL
 */
function makeHttpRequest($url, $method = 'GET', $data = null, $headers = []) {
    $ch = curl_init();
    
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, TIMEOUT);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
    
    if ($method === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        if ($data) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, is_array($data) ? json_encode($data) : $data);
        }
    }
    
    if (!empty($headers)) {
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    }
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    
    curl_close($ch);
    
    return [
        'success' => $httpCode >= 200 && $httpCode < 300,
        'code' => $httpCode,
        'response' => $response,
        'error' => $error
    ];
}

/**
 * Send JSON response
 */
function sendJsonResponse($data, $statusCode = 200) {
    http_response_code($statusCode);
    header('Content-Type: application/json');
    echo json_encode($data);
    exit;
}

/**
 * Send error response
 */
function sendError($message, $statusCode = 400) {
    sendJsonResponse([
        'success' => false,
        'message' => $message
    ], $statusCode);
}

/**
 * Send success response
 */
function sendSuccess($data, $message = 'Success') {
    sendJsonResponse([
        'success' => true,
        'message' => $message,
        'data' => $data
    ]);
}

/**
 * Clean old cache files
 */
function cleanOldCache($maxAge = 86400) {
    $cacheDir = 'cache';
    
    if (!file_exists($cacheDir)) {
        return;
    }
    
    $files = glob($cacheDir . '/*.json');
    $now = time();
    
    foreach ($files as $file) {
        if (is_file($file) && ($now - filemtime($file) > $maxAge)) {
            unlink($file);
        }
    }
}

/**
 * Clean old log files
 */
function cleanOldLogs($daysToKeep = 30) {
    $logDir = 'logs';
    
    if (!file_exists($logDir)) {
        return;
    }
    
    $files = glob($logDir . '/*.log');
    $cutoffTime = time() - ($daysToKeep * 86400);
    
    foreach ($files as $file) {
        if (is_file($file) && filemtime($file) < $cutoffTime) {
            unlink($file);
        }
    }
}

/**
 * Get file extension from URL
 */
function getFileExtension($url) {
    $path = parse_url($url, PHP_URL_PATH);
    return strtolower(pathinfo($path, PATHINFO_EXTENSION));
}

/**
 * Validate file type
 */
function isValidVideoType($extension) {
    $validTypes = ['mp4', 'webm', 'mkv', 'avi', 'mov', 'flv', 'm4v'];
    return in_array(strtolower($extension), $validTypes);
}

/**
 * Format number (views, likes, etc.)
 */
function formatNumber($number) {
    if ($number >= 1000000000) {
        return number_format($number / 1000000000, 1) . 'B';
    } elseif ($number >= 1000000) {
        return number_format($number / 1000000, 1) . 'M';
    } elseif ($number >= 1000) {
        return number_format($number / 1000, 1) . 'K';
    } else {
        return number_format($number);
    }
}

/**
 * Escape HTML output
 */
function escapeHtml($text) {
    return htmlspecialchars($text, ENT_QUOTES, 'UTF-8');
}

/**
 * Get user agent
 */
function getUserAgent() {
    return $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
}

/**
 * Check if request is from mobile device
 */
function isMobile() {
    $userAgent = getUserAgent();
    return preg_match('/(android|iphone|ipad|mobile)/i', $userAgent);
}

/**
 * Generate random string
 */
function generateRandomString($length = 16) {
    return bin2hex(random_bytes($length / 2));
}
?>
